<?php

// Include the database configuration file
include 'config.php';

// Check if the user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit();
}

// Retrieve user ID from the session
$user_id = $_SESSION['user_id'];

// Query to get user status
$status_query = "SELECT status FROM users WHERE id = :user_id";
$stmt = $pdo->prepare($status_query);
$stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
$stmt->execute();
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user || $user['status'] != 2) {
    header('Location: login.php');
    exit();
}

// Fetch statistics with IFNULL to handle potential NULL values
$queries = [
    "global_profit_query" => "SELECT IFNULL(SUM(profit), 0) AS total_profit FROM userinfo",
    "global_balance_query" => "SELECT IFNULL(SUM(balance), 0) AS total_balance FROM userinfo",
    "traders_online_query" => "SELECT COUNT(*) AS total_online FROM userinfo",
    "total_invested_query" => "SELECT IFNULL(SUM(investment), 0) AS total_invested FROM users",
    "percentage_sum_query" => "SELECT IFNULL(SUM(percentage), 0) AS total_percentage FROM users"
];

$results = [];
foreach ($queries as $key => $query) {
    $stmt = $pdo->query($query);
    $results[$key] = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Calculate Take Home Profit
$total_profit = $results['global_profit_query']['total_profit'];
$total_percentage = $results['percentage_sum_query']['total_percentage'];
$take_home_profit = ($total_percentage > 0) ? ($total_profit * $total_percentage / 100) : 0;

// Calculate Total Accumulated Profit
$total_balance = $results['global_balance_query']['total_balance'];
$total_accumulated_profit = ($total_percentage > 0) ? ($total_balance * $total_percentage / 100) : 0;

// Function to format numeric values
if (!function_exists('formatValue')) {
    function formatValue($value) {
        return is_numeric($value) ? number_format($value, 2) : '0.00';
    }
}

// Handle AJAX request
if (isset($_GET['action']) && $_GET['action'] == 'fetch_stats') {
    header('Content-Type: application/json');
    echo json_encode([
        'global_profit' => formatValue($results['global_profit_query']['total_profit']),
        'take_home_profit' => formatValue($take_home_profit),
        'global_balance' => formatValue($results['global_balance_query']['total_balance']),
        'traders_online' => number_format($results['traders_online_query']['total_online']),
        'total_invested' => formatValue($results['total_invested_query']['total_invested']),
        'total_accumulated_profit' => formatValue($total_accumulated_profit)
    ]);
    exit;
}
?>



<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Statistics</title>
    <link rel="stylesheet" href="styles.css"> <!-- Link to your stylesheet -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        /* Table styling */
        table {
            width: 90%;
            margin: 0 auto;
            border-collapse: collapse;
            background-color: rgba(0, 0, 0, 0.5); /* Transparent black background */
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.7); /* Stronger shadow for depth */
            overflow: hidden;
            color: #d4af37; /* Creamy gold text */
        }

        /* Table header and cell styling */
        td {
            padding: 15px;
            text-align: left;
            color: #d4af37; /* Creamy gold */
            font-size: 20px;
            border-bottom: 1px solid #6f6242; /* Subtle border color */
        }

        /* Alternating row colors */
        tr:nth-child(even) {
            background-color: rgba(21, 37, 37, 0.3); /* Slightly transparent dark green for even rows */
        }

        tr:nth-child(odd) {
            background-color: rgba(0, 0, 0, 0.3); /* Transparent black for odd rows */
        }

        /* Bold the first column */
        td:first-child {
            font-weight: bold;
            background-color: rgba(21, 37, 37, 0.6); /* Dark green for the first column */
            color: #b2ac8a; /* Lighter creamy gold for emphasis */
        }

        /* Format numeric values */
        td span {
            font-weight: 500;
            color: #d7d1ac; /* Soft gold */
            font-size: 20px;
        }

        /* Hover effect for rows */
        tr:hover {
            background-color: rgba(13, 64, 49, 0.5); /* Dark green highlight on hover */
            transition: background-color 0.3s ease-in-out;
        }

        /* Responsive design */
        @media (max-width: 768px) {
            table {
                width: 90%;
            }

            td {
                font-size: 14px;
                padding: 10px;
            }

            td span {
                font-size: 20px;
            }
        }
    </style>
</head>
<body>
    <br>
    <table>
        <tbody>
            <tr>
                <td>Total Accumulated Profit:</td>
                <td>£<span id="total_accumulated_profit"><?php echo formatValue($total_accumulated_profit); ?></span></td>
            </tr>
            <tr>
                <td>Total Balance:</td>
                <td>£<span id="global_balance"><?php echo formatValue($results['global_balance_query']['total_balance']); ?></span></td>
            </tr>
            <tr>
                <td>Total Online:</td>
                <td><span id="traders_online"><?php echo number_format($results['traders_online_query']['total_online']); ?></span></td>
            </tr>
            <tr>
                <td>Total Invested:</td>
                <td>£<span id="total_invested"><?php echo formatValue($results['total_invested_query']['total_invested']); ?></span></td>
            </tr>
        </tbody>
    </table>
    <br>
</body>
</html>
